<?php

namespace App\Http\Controllers\ProvinceAndCity;

use App\Http\Controllers\Controller;
use App\Models\ProvinceAndCity\CityModel;
use App\Models\ProvinceAndCity\ProvinceModel;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Facades\Excel;
use App\Imports\DataImport;
class ProvinceAndCityController extends Controller
{


    /**
     * @OA\Get(
     *     path="/v1/provinceAndCityList",
     *     summary="Get Province and City List",
     *     description="Retrieve a list of provinces and cities",
     *     tags={"Province and City"},
     *     @OA\Response(
     *         response=200,
     *         description="Successful operation",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="status", type="integer", example=1),
     *             @OA\Property(
     *                 property="data",
     *                 type="object",
     *                 description="Province and city data",
     *                 @OA\Property(property="city", type="array", @OA\Items(ref="#/components/schemas/City")),
     *                 @OA\Property(property="province", type="array", @OA\Items(ref="#/components/schemas/Province"))
     *             ),
     *             @OA\Property(property="message", type="string", example="لیست استان ها و شهر ها با موفقیت دریافت شد")
     *         )
     *     )
     * )
     *
     * @OA\Schema(
     *     schema="City",
     *     title="City",
     *     description="City model data",
     *     @OA\Property(property="id", type="integer", example=1),
     *     @OA\Property(property="name", type="string", example="City Name"),
     *     @OA\Property(property="province_id", type="integer", example=1),
     *     @OA\Property(property="isActivate", type="boolean", example=true),
     *     @OA\Property(property="created_at", type="string", format="date-time"),
     *     @OA\Property(property="updated_at", type="string", format="date-time")
     * )
     *
     * @OA\Schema(
     *     schema="Province",
     *     title="Province",
     *     description="Province model data",
     *     @OA\Property(property="id", type="integer", example=1),
     *     @OA\Property(property="name", type="string", example="Province Name"),
     *     @OA\Property(property="created_at", type="string", format="date-time"),
     *     @OA\Property(property="updated_at", type="string", format="date-time")
     * )
     */
    public function getProvinceAndCityList()
    {
        $province = ProvinceModel::select('id', 'name')->where('id', '!=', 32)->get();
        return response()->json(['status' => 1,
            'data' => $province,
            'message' => 'لیست استان ها و شهر ها با موفقیت دریافت شد']);
    }

    /**
     * @OA\Post(
     *     path="/v1/cityWithProvinceID",
     *     summary="Get City List with Province ID",
     *     description="Retrieve a list of cities based on province ID",
     *     tags={"Province and City"},
     *     @OA\Parameter(
     *         name="province_id",
     *         in="query",
     *         required=true,
     *         description="Province ID",
     *         @OA\Schema(type="integer")
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Successful operation",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="status", type="integer", example=1),
     *             @OA\Property(
     *                 property="data",
     *                 type="object",
     *                 description="City data",
     *                 @OA\Property(property="city", type="array", @OA\Items(ref="#/components/schemas/City"))
     *             ),
     *             @OA\Property(property="message", type="string", example="لیست شهر ها براساس کد استان با موفقیت دریافت شد")
     *         )
     *     ),
     *     @OA\Response(
     *         response=404,
     *         description="City list not found",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="status", type="integer", example=0),
     *             @OA\Property(property="message", type="string", example="لیست یافت نشد"),
     *             @OA\Property(property="code", type="integer", example=300)
     *         )
     *     )
     * )
     */
    public function getCityWithProvinceID(Request $request)
    {

        if (!isset($request->province_id)) {
            return response()->json(['status' => 0,
                'message' => 'کد استان برای دریافت لیست شهرهای استان لازم میباشد', 'code' => 301]);
        }

        $city = CityModel::where('province_id', $request->province_id)->select('id', 'province_id', 'uc', 'name', 'isActivate', 'merchant_id','created_at', 'updated_at')->get();

        if (count($city) > 0) {

            return response()->json(['status' => 1,
                'data' => ['city' => $city],
                'message' => 'لیست شهر ها براساس کد استان با موفقیت دریافت شد']);
        } else {

            return response()->json(['status' => 0,
                'message' => 'لیست یافت نشد', 'code' => 300]);

        }
    }

    public function getMerchantID(Request $request){
        $city = CityModel::select('merchant_id','name')->where('id', $request->id)->first();
        if ($city){
            return response()->json(['status' => 1,
                'message' => 'مرچند ایدی سازمان دریافت شد', 'data'=>['description'=>'افزایش موجودی برای شهر '. $city->name,'merchant_id'=>$city->merchant_id]]);
        } else {
            return response()->json(['status' => 0,
                'message' => 'شهر یافت نشد', 'code' => 367]);
        }
   
    }
    
    public function insertCities(Request $request){
        $request->validate([
            'file' => 'required|mimes:xlsx',
        ]);

        // Import the Excel file
        Excel::import(new DataImport, $request->file('file'));

    }
    public function getAllCity(Request $request)
    {


        $city = CityModel::select('id', 'province_id', 'uc', 'name', 'isActivate', 'merchant_id','created_at', 'updated_at')->get();

        if (count($city) > 0) {

            return response()->json(['status' => 1,
                'data' => ['city' => $city],
                'message' => 'لیست تمامی شهرهای کشور با موفقیت دریافت شد']);
        } else {

            return response()->json(['status' => 0,
                'message' => 'لیست یافت نشد', 'code' => 300]);

        }
    }
}
